## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>",
  eval = TRUE,
  warning = FALSE,
  fig.width = 7,
  fig.height = 4,
  fig.align='center'
)

## -----------------------------------------------------------------------------
# Load RuHere and terra
library(RuHere)
library(terra)

# Loading the example data
data("occurrences", package = "RuHere")

## ----set real directory, echo=FALSE-------------------------------------------
dataset_dir <- system.file("extdata", "datasets",  package = "RuHere")
spp <- unique(occurrences$species)

## ----eval = FALSE-------------------------------------------------------------
# # Define a directory for datasets (using a temporary one for this example)
# # In a real project, use a persistent path
# dataset_dir <- file.path(tempdir(), "datasets")
# dir.create(dataset_dir, showWarnings = FALSE)

## ----eval = FALSE-------------------------------------------------------------
# # Download WCVP
# wcvp_here(data_dir = dataset_dir)
# #> trying URL 'https://sftp.kew.org/pub/data-repositories/WCVP/wcvp.zip'
# #> Content type 'application/zip' length 105093449 bytes (100.2 MB)
# #> ==================================================
# #> downloaded 100.2 MB
# #>
# #> trying URL 'https://zenodo.org/records/17455838/files/wgsrpd.gpkg?download=1'
# #> Content type 'application/octet-stream' length 8581120 bytes (8.2 MB)
# #> ==================================================
# #> downloaded 8.2 MB
# #>
# #> Please don't forget to cite:
# #>
# #> Govaerts, R., Nic Lughadha, E. et al. The World Checklist of Vascular Plants, a continuously updated resource for exploring global
# #> plant diversity. Sci Data, 8, 215 (2021). https://doi.org/10.1038/s41597-021-00997-6

## ----eval = FALSE-------------------------------------------------------------
# # Set species
# spp <- unique(occurrences$species)
# spp
# #> [1] "Araucaria angustifolia"     "Cyanocorax caeruleus"       "Handroanthus serratifolius"
# 
# # Download BIEN maps for your species
# bien_here(data_dir = dataset_dir, species = spp)
# #>                       species range_available
# #>                        <char>          <lgcl>
# #> 1:     Araucaria angustifolia           FALSE
# #> 2:       Cyanocorax caeruleus           FALSE
# #> 3: Handroanthus serratifolius            TRUE

## ----eval = FALSE-------------------------------------------------------------
# # Set your key (do this once)
# set_iucn_credentials("YOUR_IUCN_KEY_HERE")

## ----eval = FALSE-------------------------------------------------------------
# # Download data for specific species
# iucn_here(data_dir = dataset_dir, species = spp)
# #> trying URL 'https://zenodo.org/records/17455838/files/wgsrpd.gpkg?download=1'
# #> Content type 'application/octet-stream' length 8581120 bytes (8.2 MB)
# #> ==================================================
# #> downloaded 8.2 MB

## ----eval = FALSE-------------------------------------------------------------
# florabr_here(data_dir = dataset_dir, data_version = "latest")
# #> Getting data from Flora e Funga do Brasil...
# #> Data will be saved in /datasets/florabr
# #>
# #> Downloading version: 393.421
# #>
# #> Merging data. Please wait a moment...
# #>
# #> Data downloaded and merged successfully. Final data saved in datasets/florabr/393.421/CompleteBrazilianFlora.rds
# #>
# #> Please don't forget to cite:
# #> Trindade WCF (2024). “florabr: An R package to explore and spatialize species distribution using Flora e Funga do
# #>  Brasil.” Applications in Plant Sciences, e11616. doi:10.1002/aps3.11616 <https://doi.org/10.1002/aps3.11616>
# >>>>>>> Stashed changes

## ----eval = FALSE-------------------------------------------------------------
# faunabr_here(data_dir = dataset_dir)
# #> Getting data from Taxonomic Catalog of the Brazilian Fauna ...
# #> Downloading version: 1.45
# #>
# #> Merging data. Please wait a moment...
# #>
# #> Data will be saved in /datasets/faunabr
# #> Data downloaded and merged successfully. Final data saved in datasets/faunabr/1.45/CompleteBrazilianFauna.gz
# 
# #> Please don't forget to cite:
# 
# #> Trindade WCF (2025) faunabr: An R package to explore taxonomic data and map species distributions using the Catálogo Taxonômico da Fauna do Brasil. Zoologia 42: e25027. <https://doi.org/10.1590/S1984-4689.v42.e25027>.

## ----eval = FALSE-------------------------------------------------------------
# fs::dir_tree(dataset_dir)
# #> datasets
# #> ├── bien
# #> │   └── Handroanthus_serratifolius.gpkg
# #> ├── faunabr
# #> │   └── 1.43
# #> │       └── CompleteBrazilianFauna.gz
# #> ├── florabr
# #> │   └── 393.420
# #> │       └── CompleteBrazilianFlora.rds
# #> ├── iucn
# #> │   └── iucn_distribution.gz
# #> ├── wcvp
# #> │   └── wcvp.gz
# #> └── wgsrpd
# #>     └── wgsrpd.gpkg

## -----------------------------------------------------------------------------
# Check availability for our species
avail <- available_datasets(data_dir = dataset_dir, species = spp)
avail

## ----out.width = "80%"--------------------------------------------------------
# Get available ranges for species
avail_ranges <- available_datasets(data_dir = dataset_dir, species = spp, 
                                   return_distribution = TRUE)
# Plot ranges of Araucaria
ranges_araucaria <- avail_ranges$species_range$`Araucaria angustifolia`

# Save current graphical parameters
oldpar <- par(no.readonly = TRUE)

# Set the plotting layout (1 row, 3 columns)
par(mfrow = c(1, 3)) #Create grid
plot(ranges_araucaria$florabr$states_biomes, main = "florabr")
plot(ranges_araucaria$iucn, main = "IUCN")
plot(ranges_araucaria$wcvp, main = "WCVP")

# Restore original graphical parameters
par(oldpar)

## -----------------------------------------------------------------------------
# Flag using WCVP-defined range
occ <- flag_wcvp(data_dir = dataset_dir, #Directory where dataset was saved
                 occ = occurrences)

## -----------------------------------------------------------------------------
# Number of records flagged (FALSE) for each species
table(occ$species, occ$wcvp_flag)

## -----------------------------------------------------------------------------
# Flag using IUCN-defined range
occ <- flag_iucn(data_dir = dataset_dir, #Directory where dataset was saved
                 occ = occ)

## -----------------------------------------------------------------------------
# Number of records flagged (FALSE) for each species
table(occ$species, occ$iucn_flag)

## -----------------------------------------------------------------------------
# Flag using BIEN-defined range
occ <- flag_bien(data_dir = dataset_dir, #Directory where dataset was saved
                 occ = occ)
#> Checking the distribution from 1 of 3 species

## -----------------------------------------------------------------------------
# Number of records flagged (FALSE) for each species
table(occ$species, occ$bien_flag)

## -----------------------------------------------------------------------------
# Flag using florabr-defined range
occ <- flag_florabr(data_dir = dataset_dir, #Directory where dataset was saved
                    occ = occ)

## -----------------------------------------------------------------------------
# Number of records flagged (FALSE) for each species
table(occ$species, occ$florabr_flag)

## -----------------------------------------------------------------------------
# Flag using faunabr-defined range
occ <- flag_faunabr(data_dir = dataset_dir, #Directory where dataset was saved
                    occ = occ)

## -----------------------------------------------------------------------------
# Number of records flagged (FALSE) for each species
table(occ$species, occ$faunabr_flag)

## ----eval=FALSE---------------------------------------------------------------
# # Interactive map with map_here()
# map_here(occ, species = "Araucaria angustifolia", label = "record_id", cex = 4)

## ----show mapview, eval=T, echo=F, fig.align='center', out.width='80%'--------
knitr::include_graphics("vignettes_img/IMG03.jpeg")

## ----out.width = "80%"--------------------------------------------------------
ggmap_here(occ, species = "Araucaria angustifolia", facet_wrap = TRUE)

## -----------------------------------------------------------------------------
occ_cleaned <- remove_flagged(occ)

## -----------------------------------------------------------------------------
flag_summary <- summarize_flags(occ)

## -----------------------------------------------------------------------------
# Data.frame summarizing the number of records per flag
flag_summary$df_summary

## ----out.width = "80%"--------------------------------------------------------
# Bar plot
flag_summary$plot_summary

